function hs(original_file,target_file,random_approach)
% HS - Optimum exact histogram specification, and inverse (reconstruction)
%   
%      The image in original_file is given the histogram of the image in
%      target_file (both images have the same number of pixels and bpp
%      and be grayscale)
%
%      If only original_file is given, or if target_file='', then the
%      target histogram is assumed to be flat (i.e., histogram equalisation)
%
%      random_approach=1 uses the random approach for reconstruction
%      (if argument not given, reconstruction uses stable approach
%       i.e., random_approach=0)

% Copyright (C) 2016 Félix Balado
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.  
  
% $Id: hs.m,v 1.2 2017/01/09 15:28:39 felix Exp $
  
  %% a couple of minor parameters can be changed here
  draw=1; % display figures or not?

  if nargin<3
    random_approach=0;
  end
  
  close all;

  %% read original image
  im_z=imread(sprintf('images/%s',original_file)); % Mila Nikolova's images
  im_z_info=imfinfo(sprintf('images/%s',original_file));
  fprintf('Original: %s (%ix%i), %i bpp\n',original_file,im_z_info.Width,im_z_info.Height,im_z_info.BitDepth);
  
  n=im_z_info.Width*im_z_info.Height;

  % concatenate image rows
  im_z=im_z';
  z=double(im_z(:));
  im_z=im_z';
  
  v=[0:2^im_z_info.BitDepth-1]; % handier than v=unique(z) in this problem;
                                % just take into account that some histogram 
                                % bins can be zero
  q=length(v);
  hz=histc(z,v);

  if nargin>1 && ~isempty(target_file) % as we can specify ''
    %% read target
    im_x=imread(sprintf('images/%s',target_file));
    x=double(im_x(:));
    if length(x)~=n        
      error('target image must have same number of pixels as original image')
    end
    fprintf('Target: %s\n',target_file);

    hx=histc(x,v);
    xs=sort(x,'ascend');
  else
    %% histogram equalisation (no need to read x)
    hx=floor(n/q)*ones(q,1);
    if (n/q)~=floor(n/q)
      fprintf('completely flat histogram not possible, fixing\n');
      w=n-sum(hx);
      hx(1:w)=hx(1:w)+1; % adjust so that sum(hx)=n;
      
      %% random adjustment (alternative to adding 1 to first w bins as above)
      % fix=[ones(1,w) zeros(1,q-w)]';
      % idx=randperm(q);
      % hx=hx+fix(idx);
    end
    x=[];
    for k=1:q
      x=[x; v(k)*ones(hx(k),1)];
    end
    xs=x;
  end

  %% find closest equalization using minimum distance decoder
  [zs,indexz]=sort(z,'ascend'); % find Πz using stable sorting
  
  y(indexz)=xs; % y=(Πz')xs 
  y=y(:);
  
  hy=histc(y,v);
  im_y=uint8(reshape(y(:),size(im_z,1),size(im_z,2)));
  im_y=im_y';
  
  %% RECONSTRUCTION

  %% get H matrix
  % Lambda_xs=sparse(zeros(n,q));
  % Lambda_zs=sparse(zeros(n,q));
  % for k=1:q
  %   Lambda_xs(:,k)=(xs==v(k));
  %   Lambda_zs(:,k)=(zs==v(k));
  % end
  % H=Lambda_zs'*Lambda_xs;
  
  %% get H matrix, but much faster and using a lot less memory
  H=zeros(q,q);
  w=1;
  for k=1:q
    if hz(k)>0
      H(k,:)=histc(xs(w:w+hz(k)-1),v);
      w=w+hz(k);
    end
  end
  
  if random_approach
    r_mode='random';
    [dummy,indexy]=randomsort(y,'ascend');
  else
    r_mode='stable';
    [dummy,indexy]=sort(y,'ascend');

   %% in general, indexy is not increasing within equal values of z 
   %% and then we are not recovering stable sorting; the following is 
   %% the heuristic stable approach described in the paper 
    w=1;
    for k=1:q
      if hx(k)>0
    	indexy(w:w+hx(k)-1)=indexy(w+hx(k)-1:-1:w);  % reverse sorted vector
    	w=w+hx(k);
      end
    end  

    %% the following bit can actually be skipped as it only involves
    %% ties of zs, although it is formally the way to recover stable sorting
    w=1;
    for k=1:q
      if hz(k)>0
    	indexy(w:w+hz(k)-1)=sort(indexy(w:w+hz(k)-1),'ascend');
    	w=w+hz(k);
      end
    end
  end
  
  zr(indexy)=zs;  % ẑ=(Πy')zs
  hzr=histc(zr(:),v);
  im_zr=uint8(reshape(zr(:),size(im_z,1),size(im_z,2)));
  im_zr=im_zr';


  % verify histograms of equalised image and reconstruction
  % (just for fun, because they must be equal by construction)
  if sum(hx~=hy)>0
     fprintf('error, equalised histogram different from target!\n\n');
  end
  if sum(hzr~=hz)>0
     fprintf('error, reconstuction histogram different from original!\n\n');
  end
  
  %% EHS performance
  
  % empirical PSNR
  mse=mean((y(:)-z(:)).^2);
  psnr=10*log10(255^2/mse);
  
  % covering sphere
  c=mean(x)*ones(size(x));
  Rc=sqrt(norm(x)^2-(1/n)*sum(x)^2);

  %% old bounds, based on the ensemble of all rearrangements of x
  % lower bounds on \|\mathbf{z}-\mathbf{y}\|
  lb1=abs(norm(z)-norm(x));
  lb2=abs(norm(z-c)-Rc);
  lb3=abs(sum(z-x)/sqrt(n));
  l=max([lb1,lb2,lb3].^2)/n;
  lb_old=10*log10(255^2/l);

  % upper bound on \|\mathbf{z}-\mathbf{y}\|^2
  u=(norm(z)^2+norm(x)^2-(2/n)*sum(x)*sum(z))/n;
  ub_old=10*log10(255^2/u);

  % triangle inequality using all codewords (always worse), so we ignore it
  u2=(norm(z-c)+Rc)^2/n;
  ub_old_ti=10*log10(255^2/u2);

  %% new bounds, based on the optimum rearrangements only
  lb2=0;
  R2=norm(x)^2;
  w=1;
  for k=1:q
    if hz(k)>0
      pt=xs(w:w+hz(k)-1);
      lb2=lb2+hz(k)*(v(k)-sum(pt)/hz(k))^2;
      R2=R2-sum(pt)^2/hz(k);
      w=w+hz(k);
    end	  

  end
  ub2=(sqrt(lb2)+sqrt(R2))^2; % triangle inequality using optimum solutions
  
  psnr_ub_new=10*log10(255^2/(lb2/n));
  psnr_lb_new=10*log10(255^2/(ub2/n));

  fprintf(sprintf('Optimum EHS:\n'));
  fprintf(sprintf('PSNR: %.3f dB (lb=%.3f, ub=%.3f)\n',psnr,psnr_lb_new,psnr_ub_new));
  
  %% reconstruction performance
  
  % empirical PSNR  and degree of reconstruction error
  mse=mean((zr(:)-z(:)).^2);
  psnr_r=10*log10(255^2/mse);
  nu_r=mean(z(:)~=zr(:));

  mse_ub=norm(z)^2;
  mse_av=norm(z)^2;
  nu_r_av=0;
  w=1;
  for k=1:q
    if hx(k)>0
      pt=zs(w:w+hx(k)-1);
      nu_r_av=nu_r_av+(hx(k)/n)*(1-(norm(H(:,k))/hx(k))^2);
      mse_ub=mse_ub-pt'*pt(end:-1:1);
      mse_av=mse_av-sum(pt)^2/hx(k);
      w=w+hx(k);
    end
  end

  mse_ub=(2*mse_ub)/n;
  mse_av=(2*mse_av)/n;
  psnr_r_lb=10*log10(255^2/mse_ub);
  psnr_r_av=10*log10(255^2/mse_av);
  
  fprintf(sprintf('\nReconstruction (%s):\n',r_mode));
  fprintf(sprintf('PSNR: %.3f dB (av=%.3f, lb=%.3f)\n',psnr_r,psnr_r_av,psnr_r_lb));
  fprintf(sprintf('ν: %.3f (av=%.3f))\n\n',nu_r,nu_r_av));

  % pseudo LaTeX output (for table of results in paper)
%  fprintf(sprintf('& %s & %.2f & %.2f & %.2f & & %.2f & & %.2f & %.2f & %.2f & & %.2f\n\n',original_file,psnr,psnr_lb_new,psnr_ub_new,psnr_r,psnr_r_av,psnr_r_lb,nu_r,nu_r_av));
  
% display results 
  if draw
    hh=figure(1);
    subplot_tight(2,3,1);
    imshow(uint8(im_z));
    title('original');
    subplot_tight(2,3,2);
    imshow(uint8(im_y));
    title('equalised');
    subplot_tight(2,3,4);
    imshow(uint8(im_zr));
    title(sprintf('reconstruction (%s), PSNR=%.2f dB [av %.2f dB, lb=%.2f dB]',r_mode,psnr_r,psnr_r_av,psnr_r_lb));
    subplot_tight(2,3,5);
    imshow(uint8(255*(im_z==im_zr)));
    title(sprintf('reconstruction errors (black), rate=%.3f [av %.3f]',nu_r,nu_r_av));
    subplot_tight(2,3,6);
    plot(v,hz,'b--',v,hzr,'b:',v,hx,'r--',v,hy,'r:');
    legend('original','reconstruction','target','equalised');
    
    if nargin>1 && ~isempty(target_file)
      subplot_tight(2,3,3);
      imshow(im_x);
      title('target');
    end
    
    set(gcf,'Units','normalized','Position',[0,0,1,1]);

    if nargin>1 && ~isempty(target_file)
      set(gcf,'name',sprintf('%s %ix%i, %i bpp (target: %s)\n\n',original_file,im_z_info.Width,im_z_info.Height,im_z_info.BitDepth,target_file),'numbertitle','off');
    else
      set(gcf,'name',sprintf('%s %ix%i, %i bpp\n\n',original_file,im_z_info.Width,im_z_info.Height,im_z_info.BitDepth),'numbertitle','off');
    end
    
    fprintf('[close figure to continue...]\n\n');
    waitfor(hh);
  end

  
function [s,idx]=randomsort(m,dir)
%  RAMDOMSORT - (only for integer vectors m)
%               same as sort, but ties are randomly orderered 
%               in idx, rather than in the same order as in the 
%               original vector m
%               (which is called stable sorting)

  if nargin==1
    dir='ascend';
  end
  [s,idx]=sort(m+(rand(size(m))-0.5),dir);
  s=round(s); %back to integers

  
