function [lgr,ub_lgr]=log2_multinomial(h,verbose)
% LOG2_MULTINOMIAL - Compute log2 of multinomial coefficient r exactly,
%                    or lower bound it, for use in permutation encoding;
%                    the entropy upper bound on log2(r) is also returned if
%                    the computation is not exact
%   
%                    h - values in denominator of multinomial quotient

% Copyright (C) 2014 Félix Balado and David Haughton
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.  
  
% $Id: log2_multinomial.m,v 1.6 2014/10/17 10:56:58 felix Exp $
    
    
    if nargin==1
        verbose=0;
    end

    if length(h)==1 % just one type of element
        lgr=0;
	ub_lgr=0;
        return;
    end
    
    n=sum(h); % value in numerator of multinomial quotient
    
    if n<=20
        %% For small n we give log2 of multinomial coefficient using factorials
	%% factorial can only be exact up to n=20 with 64 bit precision
	
        lgr=log2(factorial(n))-sum(log2(factorial(h)));
	ub_lgr=lgr;
        if verbose
            fprintf(1,'log2-multinomial: '); 
        end

        if lgr<0 
            fprintf(1,'%3g bits (exact)\n',lgr);
        end

    else
      %% For larger n we give instead a lower bound
      %% of log(r) using a lower bound on n! and upper bounds on h_k! 
      %% it is fundamental not to overestimate log2(r) because the arithmetic 
      %% coder could never handle more than r rearrangements!

      %% Robbins' bounds used to lower bound multinomial:
      %% sqrt(2*pi*n)(n/e)^n e^(1/(12*n+1)) < n! < sqrt(2*pi*n)(n/e)^n e^(1/(12*n))
      %% Note: we cannot just use approximation \rho\approx H(X) because this
      %% yields an upper bound on log(r), and so we risk having more 
      %% messages than rearrangements even after rounding down
      
        lgr=(log(sqrt(2*pi*n))+n*log(n)-n+(1/(12*n))-sum(log(sqrt(2*pi*h))+h.*log(h)-h+(1./(12*h+1))))/log(2);
        if verbose
            fprintf(1,'Lower bound of log2-multinomial: '); 
        end
        
	p=h/n;
	nz=(p>0);
	H=-sum(p(nz).*log2(p(nz))); % upper bound on (1/n) \log2 r
	ub_lgr=n*H;
    end

    
    if verbose
        fprintf(1,'%3g bits\n',lgr);
    end
