function m=permutation_decoding_m(y,nr_bits)
% PERMUTATION_DECODING_M - Ranks (decodes) permutation.
%    M=PERMUTATION_DECODING_M(Y,NBITS) returns the rank of Y as a
%    bit sequence M of length NBITS                       
%                       
%    In steganography: NBITS=FLOOR(LOG2_MULTINOMIAL(H)), where H=HISTC(Y,V) and
%    V=UNIQUE(Y), in order to guarantee unique decoding.
%
%    Arithmetic encoder follows "Introduction to Data Compression" 
%    by K. Sayood
%  
%    Note: PERMUTATION_DECODING works just as PERMUTATION_DECODING_M but 
%    orders of magnitude faster, since it is a MEX file programmed in C
%    rather than MATLAB code.
%
%    See also: PERMUTATION_ENCODING_M, LOG2_MULTINOMIAL.

% Copyright (C) 2014 Félix Balado and David Haughton
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.  
  
% $Id: permutation_decoding_m.m,v 1.6 2014/10/17 10:49:17 felix Exp $
    
    v=unique(y); % unique values in x, sorted 
    h=histc(y,v); % histogram 
    
    % note: a secret key can be k=randperm(n), then v=v(k) and h=h(k)
    
    % lower and higher ends of initial [0,1) interval, represented by 32 bits
    low=uint32(0);         % 0.low
    high=intmax('uint32'); % 0.high
    
    m=[]; % message vector to be decoded (initially empty)

    underflow=0;
    for k=1:length(y);
        % subinterval to symbol index mapping (for available symbols)
        sub2sym=find(h>0);

        % cumulative sum of available symbol counts
        hs=h(sub2sym);
        hs=hs(:)';
        c=[0 cumsum(hs)];

        % divide [low,high) interval into q subintervals proportional to h
        range=high-low+1;
        subs=low+uint32(floor(double(range)*c/max(c)));
                 
        % the k-th watermark symbol y(k) establishes the right subinterval
        sym_index=find(y(k)==v);
        h(sym_index)=h(sym_index)-1; % decrease histogram for that symbol 
        
        % find the subinterval index corresponding to that symbol
        sub_index=find(sym_index==sub2sym); 
        
        % new interval before next step
        new_low=subs(sub_index);
        new_high=subs(sub_index+1)-1;

        while (bitget(new_low,32)==bitget(new_high,32))||((bitget(new_low,32)~=bitget(new_high,32))&&(bitget(new_low,31)==1)&&(bitget(new_high,31)==0))
            
            if (bitget(new_low,32)==bitget(new_high,32))
               
                % message bit decoded (compressed bit)
                b=bitget(new_low,32);
                m=[m b];
            
                % shifts
                new_low=bitshift(new_low,1);
                new_high=bitshift(new_high,1);
                new_high=bitset(new_high,1,1);
                
                % flush underflow bits if needed
                if underflow>0
                    m=[m uint32(repmat(~b,1,underflow))];
                    underflow=0;
                end
            end
        
            if ((bitget(new_low,32)~=bitget(new_high,32))&&(bitget(new_low,31)==1)&&(bitget(new_high,31)==0))
                
                underflow=underflow+1;
                
                % shifts
                new_low=bitshift(new_low,1);
                new_high=bitshift(new_high,1);
                new_high=bitset(new_high,1,1); % lsb
                
                % complements
                new_low=bitset(new_low,32,0);
                new_high=bitset(new_high,32,1);
            end
        end
 
        % update interval
        low=new_low;
        high=new_high;
    end
 
    b=bitget(new_high,1:32);
    % termination
    if underflow==0
        m=[m uint32(b(32:-1:1))]; 
    else
        m=[m uint32(b(32)) uint32(repmat(~b(32),1,underflow)) uint32(b(31:-1:1))];
    end

    m=m(1:nr_bits);
