#include <mex.h>
#include "math.h"
#include "inttypes.h"

/* Arithmetic decoder follows "Introduction to Data Compression"  */
/* by K. Sayood */

/* Note: C version of PERMUTATION_ENCODING_M.M  */

/* Copyright (C) 2014 Félix Balado and David Haughton */

/* This program is free software: you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation, either version 3 of the License, or */
/* (at your option) any later version. */

/* This program is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with this program.  If not, see <http://www.gnu.org/licenses/>.   */

/* $Id: permutation_encoding.c,v 1.6 2014/10/17 10:49:20 felix Exp $ */


void mexFunction(int nlhs, mxArray *plhs[],
		 int nrhs, const mxArray *prhs[])
{

  uint32_t q,q0,n,k,i,j,r,ml,t;
  uint32_t sym_index,sub_index,b;
  double *x,*v,*h,*m,*out,*y;
  uint32_t *c,*hs,*subs,*sub2sym;
  
  mxArray *xx[1],*hh[1],*vv[1],*par[2];

  /* lower and higher ends of initial [0,1) interval, represented by 32 bits*/
  uint32_t low=0;
  uint32_t high=(uint32_t) -1; /* 2^{32}-1 */

  uint32_t bit32=(1<<31);
  uint32_t bit31=(1<<30);

  uint32_t new_low,new_high;
  double range;
  uint32_t m_32;

  /* check arguments */
  if (nrhs != 2) 
    mexErrMsgTxt("Two inputs required");
  else if (nlhs > 1)
    mexErrMsgTxt("Too many output arguments");

  /* get pointers to arrays */
  x = mxGetPr(prhs[0]); 
  n = mxGetM(prhs[0])*mxGetN(prhs[0]); /* vector length */

  if(!mxIsClass(prhs[1],"double"))
    {
      mexErrMsgTxt("bitstream must be of type double\n");
    }
  m = mxGetPr(prhs[1]); 
  ml = mxGetM(prhs[1])*mxGetN(prhs[1]); /* message length in bits */

  xx[0]=prhs[0];
  mexCallMATLAB(1,vv,1,xx,"unique"); 
  q = mxGetM(vv[0])*mxGetN(vv[0]);
  q0 = q;
  v=mxGetPr(vv[0]);
  
  par[0]=prhs[0];
  par[1]=vv[0];
  mexCallMATLAB(1,hh,2,par,"histc"); 
  h=mxGetPr(hh[0]);

  sub2sym=mxCalloc(q,sizeof(uint32_t));
  hs=mxCalloc(q,sizeof(uint32_t));
  c=mxCalloc(q+1,sizeof(uint32_t));
  subs=mxCalloc(q+1,sizeof(uint32_t));

  /* vector to be encoded */
  y=mxCalloc(n,sizeof(double));

  m_32=0;
  i=0;  /* encoded bit index */
  while(i<32)
    {
      if(i<ml)
	m_32=(m_32|(uint32_t)m[i]); /* otherwise pad with zeroes */
      if(i<31)
	m_32=(m_32<<1);
      i++;
    }
  
  for(k=0;k<n;k++)
    {
      /* subinterval to symbol index mapping (for available symbols)*/
      r=0;
      for(j=0;j<q0;j++)
	if(h[j]>0)
	  {
	    sub2sym[r]=j;
	    hs[r]=(uint32_t)h[j];
	    r++;
	  }
      q=r;
      
      /* cumulative sum of available symbol counts*/
      c[0]=0;
      for(j=0;j<q;j++)
	c[j+1]=c[j]+hs[j];

      /* divide [low,high) interval into q subintervals proportional to h*/
      range=(double)high-(double)low+1;

      for(j=0;j<=q;j++)
	subs[j]=low+(uint32_t)floor(range*(double)c[j]/(double)c[q]);
      
      /* find subinterval were m_32 lies*/
      t=(uint32_t)floor((((m_32-low+1)*(double)c[q])-1)/range);
      sub_index=0;
      while(t>=c[sub_index])
	sub_index++;
      sub_index--;
      sym_index=sub2sym[sub_index]; /* corresponding symbol index */

      /* subinterval establishes k-th watermarked signal symbol y[k]*/
      y[k]=v[sym_index];
      h[sym_index]=h[sym_index]-1; /* decrease histogram for that symbol */

      /* new interval before next step*/
      new_low=subs[sub_index];
      new_high=subs[sub_index+1]-1;

      while(((new_low&bit32)==(new_high&bit32))||
	    (((new_low&bit32)!=(new_high&bit32))&&((new_low&bit31)!=0)&&((new_high&bit31)==0)))
	{            
	  if((new_low&bit32)==(new_high&bit32))
	    {   
	      /* shifts*/
	      new_low=(new_low<<1);
	      new_high=(new_high<<1);
	      new_high=(new_high|(uint32_t)1); /* lsb*/
              
	      m_32=(m_32<<1);
	      b=(i<ml?(uint32_t)m[i]:0);
	      i++;
	      m_32=(m_32|b);
	    }
             
	  if(((new_low&bit32)!=(new_high&bit32))&&((new_low&bit31)!=0)&&((new_high&bit31)==0))  
	    {
	      /* shifts*/
	      new_low=(new_low<<1);
	      new_high=(new_high<<1);
	      new_high=(new_high|(uint32_t)1);	      
	      
	      m_32=(m_32<<1);
	      b=(i<ml?(uint32_t)m[i]:0);
	      i++;
	      m_32=(m_32|b);
	      
	      /* complements*/
	      new_low=(new_low&(~bit32));
	      new_high=(new_high|bit32);
	      b=(uint32_t)((m_32&bit32)!=0);
	      if(b==0)
		m_32=(m_32|bit32);
	      else
		m_32=(m_32&(~bit32));
	    }
        }
      
        /* update interval*/
        low=new_low;
        high=new_high;
    }

  plhs[0] = mxCreateDoubleMatrix(1,n,mxREAL);
  out = mxGetPr(plhs[0]);
  for (j=0;j<n;j++) 
    out[j]=y[j];

  mxFree(y);
  mxFree(subs);
  mxFree(c);
  mxFree(hs);
  mxFree(sub2sym);
}
