function y=permutation_encoding_m(x,m)
% PERMUTATION_ENCODING_M - Unranks (encodes) permutation.
%    Y=PERMUTATION_ENCODING_M(X,M) returns the permutation Y with
%    histogram H=HISTC(X,V), with V=UNIQUE(X), whose rank is given by the 
%    bit sequence M.
%                       
%    In steganography: NBITS=LENGTH(M) must be <= LOG2_MULTINOMIAL(H) in 
%    order to guarantee unique decoding.
%  
%    Arithmetic decoder follows "Introduction to Data Compression" 
%    by K. Sayood
%  
%    Note: PERMUTATION_ENCODING works just as PERMUTATION_ENCODING_M but 
%    orders of magnitude faster, since it is a MEX file programmed in C
%    rather than MATLAB code.
%
%    See also: PERMUTATION_DECODING_M, LOG2_MULTINOMIAL.
  
% Copyright (C) 2014 Félix Balado and David Haughton
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.  
  
% $Id: permutation_encoding_m.m,v 1.5 2014/10/17 10:48:57 felix Exp $
    
    v=unique(x); % unique values in x, sorted 
    h=histc(x,v); % histogram 
    
    % note: a secret key can be k=randperm(length(v)), then v=v(k) and h=h(k)
    
    y=zeros(size(x)); % watermarked vector
    
    % lower and higher ends of initial [0,1) interval, represented by 32 bits
    low=uint32(0);         % 0.low
    high=intmax('uint32'); % 0.high
    
    % get 32-bit message chunk 
    m_32=uint32(0);
    i=1;
    while i<=32
        b=get_message_bit(m,i); 
        m_32=bitset(m_32,33-i,b);
        i=i+1;
    end

    for k=1:length(x);

        % subinterval to symbol index mapping (for available symbols)
        sub2sym=find(h>0);
        
        % cumulative sum of available symbol counts
        hs=h(sub2sym);
        hs=hs(:)';
        c=[0 cumsum(hs)];

        % divide [low,high) interval into q subintervals proportional to h
        range=high-low+1;
        subs=low+uint32(floor(double(range)*c/max(c)));
        
        % find subinterval were m_32 lies
        sub_index=1;
        while(floor((double(m_32-low+1)*max(c)-1)/double(range))>=c(sub_index))
            sub_index=sub_index+1;
        end
        sub_index=sub_index-1;
        sym_index=sub2sym(sub_index); % corresponding symbol index 

        % subinterval establishes k-th watermark symbol y(k)
        y(k)=v(sym_index);
        h(sym_index)=h(sym_index)-1; % decrease histogram for that symbol 

        % new interval before next step
        new_low=subs(sub_index);
        new_high=subs(sub_index+1)-1;
         
        while (bitget(new_low,32)==bitget(new_high,32))||((bitget(new_low,32)~=bitget(new_high,32))&&(bitget(new_low,31)==1)&&(bitget(new_high,31)==0))
            
            if (bitget(new_low,32)==bitget(new_high,32))
                
                % shifts
                new_low=bitshift(new_low,1);
                new_high=bitshift(new_high,1);
                new_high=bitset(new_high,1,1); % lsb
                
                m_32=bitshift(m_32,1);
                b=get_message_bit(m,i);
                i=i+1;
                m_32=bitset(m_32,1,b);
            end
             
            if ((bitget(new_low,32)~=bitget(new_high,32))&&(bitget(new_low,31)==1)&&(bitget(new_high,31)==0))   

                % shifts
                new_low=bitshift(new_low,1);
                new_high=bitshift(new_high,1);
                new_high=bitset(new_high,1,1);

                m_32=bitshift(m_32,1);
                b=get_message_bit(m,i);                
                i=i+1;
                m_32=bitset(m_32,1,b);
 
                % complements
                new_low=bitset(new_low,32,0);
                new_high=bitset(new_high,32,1);
                b=bitget(m_32,32);
                m_32=bitset(m_32,32,~b);
            end
        end
        
        % update interval
        low=new_low;
        high=new_high;
    end
    
function b=get_message_bit(m,i)
% GET_MESSAGE_BIT - 
%   
    
    if i>length(m)
        b=0; % pad with zeros
    else
        b=m(i);
    end